#!/usr/bin/env node

import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js"
import { StreamableHTTPServerTransport } from "@modelcontextprotocol/sdk/server/streamableHttp.js"
import { createMcpExpressApp } from "@modelcontextprotocol/sdk/server/express.js"
import pkg from "../manifest.json" with { type: "json" }


const app = createMcpExpressApp()

app.post("/mcp", async (req, res) => {
	const server = getServer()
	try {
		const transport = new StreamableHTTPServerTransport({
			sessionIdGenerator: undefined,
		})
		await server.connect(transport)
		await transport.handleRequest(req, res, req.body)
		res.on("close", () => {
			transport.close()
			server.close()
		})
	} catch (error) {
		console.error("Error handling MCP request:", error)
		if (!res.headersSent) {
			res.status(500).json({
				jsonrpc: "2.0",
				error: {
					code: -32603,
					message: "Internal server error"
				},
				id: null
			})
		}
	}
})

//         监听端口 0，随机选择端口
app.listen(0, function (err) {
	if (err) {
		console.error("Failed to start server:", err)
		process.exit(1)
	}
	// 输出监听的端口号
	console.log(JSON.stringify({
		"type": "http_start",
		"port": this.address().port
	}))
})

/**
 * 初始化MCP Server
 * @returns {McpServer}
 */
function getServer() {
	const server = new McpServer(
		{ name: pkg.name, version: pkg.version },
		{
			//             申明服务有Prompt扩展能力
			capabilities: { prompts: {}, tools: {} },
		}
	)

	// 添加一个空工具避免tools/list报错
	server.registerTool("_", {}, () => {})

	// 系统提示词
	server.registerPrompt(
		"addition-system-instruction",
		{
			description: "document generation prompt",
		},
		() => {
			return {
				"description": "document generation prompt",
				"messages": [
					{
						"role": "user",
						"content": {
							"type": "text",
							"text": `
Document Generation Instructions:
You are a document generation AI. Your task is to convert the user's input into a well-structured document format. Follow these guidelines:

1. Understand the Content:
	 - Carefully read and comprehend the user's input to grasp the main ideas and themes.
2. Structure the Document:
	 - Organize the content into sections with clear headings and subheadings.
	 - Use bullet points or numbered lists for clarity where appropriate.
3. Formatting:
	 - Ensure consistent formatting throughout the document.
	 - Use bold or italics to emphasize key points.
4. Clarity and Conciseness:
	 - Write in a clear and concise manner, avoiding unnecessary jargon.
	 - Ensure that each section flows logically to the next.
5. Review:
	 - Double-check for any grammatical errors or typos.
	 - Ensure that the final document meets the user's requirements.

Please generate the document based on these instructions.
`.trimStart()
						}
					}
				]
			}
		}
	)

	return server
}

process.on("SIGINT", async () => {
	console.log("Server shutdown complete")
	process.exit(0)
})
